package fr.asip.cps3.exemple.modele.objets;

import iaik.pkcs.pkcs11.wrapper.CK_TOKEN_INFO;
import iaik.pkcs.pkcs11.wrapper.PKCS11;
import iaik.pkcs.pkcs11.wrapper.PKCS11Constants;
import iaik.pkcs.pkcs11.wrapper.PKCS11Exception;

import org.apache.log4j.Logger;

import fr.asip.cps3.exemple.modele.exceptions.ExceptionObjet;
import fr.asip.cps3.exemple.modele.exceptions.ExceptionProgrammeExemple;
import fr.asip.cps3.exemple.modele.traitements.util.TraitementsUtil;

/**
 * Classe modlisant un token(carte)
 *
 */
public class Token {

	/**
	 * Le loggeur
	 */
	 private static Logger log = Logger.getLogger(Token.class);
	
	/**
	 * Identifiant du slot hbergant le token
	 */
	private long idSlotToken = -1;	

	/**
	 * Informations sur le token
	 */
	private CK_TOKEN_INFO info;
	
	/**
	 * Constructeur
	 * @param librairie Librairie PKCS#11  utiliser
	 * @param idSlotToken Identifiant du slot hbergant le token
	 * @throws ExceptionObjet 
	 */
	public Token(PKCS11 librairie, long idSlotToken) throws ExceptionObjet {
		
		this.idSlotToken = idSlotToken;
		
		chargeInfo(librairie);
	}

	/**
	 * (Re)chargement explicite des informations du token
	 * @param librairie Librairie PKCS#11  utiliser
	 * @throws ExceptionObjet 
	 * @throws ExceptionObjet
	 */
	public void chargeInfo(PKCS11 librairie) throws ExceptionObjet{
		
		if(librairie != null) {
			
			try {
			
				info = librairie.C_GetTokenInfo(idSlotToken);
				
			} catch (PKCS11Exception e) {

				// Si une erreur PKCS#11 est rencontre on log et on lve l'exception
				log.error("Une erreur est survenue lors de la recuperation des informations de la carte : "+TraitementsUtil.retranscritCodeRetour(e.getErrorCode()));
				throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Une erreur est survenue lors de la rcuperation des informations de la carte");
				
			}
			
		} else {
		
			// Si la librairie est nulle il est impossible de rcuprer les informations de la carte
			log.error("La librairie est nulle");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "La librairie est nulle");
			
		}
		
	}
	
	/**
	 * Renvoie le libell du token
	 * @return Libell du token
	 */
	public String getLibelle() {
		
		if(info != null && idSlotToken != -1)
			return new String(info.label).trim()+" (Slot "+idSlotToken+")";
		return "Carte "+Long.toString(idSlotToken);
		
	}

	/**
	 * Accesseur
	 * @return Identifiant du slot hbergant le token
	 */
	public long getIdSlotToken() {
		return idSlotToken;
	}

	/**
	 * Renvoie le libell application attribu lors de l'initialisation de la carte
	 * @return Libell application attribu lors de l'initialisation de la carte
	 * @throws ExceptionObjet 
	 */
	public String getLibelleInit() throws ExceptionObjet {
		
		if(info != null)
			return new String(info.label).trim();
		else {
			
			log.error("Les informations de la carte sont nulles, le libell application attribu lors de l'initialisation de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le libell application attribu lors de l'initialisation de la carte est indisponible");
			
		}
		
	}	

	/**
	 * Renvoie l'identifiant du fabricant de la carte
	 * @return Identifiant du fabricant de la carte
	 * @throws ExceptionObjet 
	 */
	public String getFabricant() throws ExceptionObjet {
		
		if(info != null)
			return new String(info.manufacturerID);
		else {
			
			log.error("Les informations de la carte sont nulles, l'identifiant du fabricant de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'identifiant du fabricant de la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie le modle de la carte
	 * @return Modle de la carte
	 * @throws ExceptionObjet 
	 */
	public String getModele() throws ExceptionObjet {
		
		if(info != null)
			return new String(info.model);
		else {
			
			log.error("Les informations de la carte sont nulles, le modele de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le modle de la carte est indisponible");
			
		}
	}
	
	/**
	 * Renvoie le numro de srie de la carte
	 * @return Numro de srie de la carte
	 * @throws ExceptionObjet 
	 */
	public String getNumeroSerie() throws ExceptionObjet {
		
		if(info != null)
			return new String(info.serialNumber);
		else {
			
			log.error("Les informations de la carte sont nulles, le numero de serie de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le numro de srie de la carte est indisponible");
			
		}
	}	
	
	/**
	 * Renvoie le nombre maximum de sessions pouvant tre ouvertes simultanment par une mme application sur la carte
	 * @return Nombre maximum de sessions pouvant tre ouvertes simultanment par une mme application sur la carte
	 * @throws ExceptionObjet 
	 */
	public long getMaxSessionsSimultanees() throws ExceptionObjet {
		
		if(info != null)
			return info.ulMaxSessionCount;
		else {
			
			log.error("Les informations de la carte sont nulles, le nombre maximum de sessions pouvant etre ouvertes simultanement par une meme application sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le nombre maximum de sessions pouvant tre ouvertes simultanment par une mme application sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie le nombre de sessions que l'application courante a d'ouvertes sur la carte
	 * @return Nombre de sessions que l'application courante a d'ouvertes sur la carte
	 * @throws ExceptionObjet 
	 */
	public long getNombreSessionsApplicationCourante() throws ExceptionObjet {
		
		if(info != null)
			return info.ulSessionCount;
		else {
			
			log.error("Les informations de la carte sont nulles, le nombre de sessions que l'application courante a d'ouvertes sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le nombre de sessions que l'application courante a d'ouvertes sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie le nombre maximum de sessions en lecture/criture pouvant tre ouvertes simultanment par une mme application sur la carte
	 * @return Nombre maximum de sessions en lecture/criture pouvant tre ouvertes simultanment par une mme application sur la carte
	 * @throws ExceptionObjet 
	 */
	public long getMaxSessionsSimultaneesEnEcriture() throws ExceptionObjet {
		
		if(info != null)
			return info.ulMaxRwSessionCount;
		else {
			
			log.error("Les informations de la carte sont nulles, le nombre maximum de sessions en lecture/ecriture pouvant etre ouvertes simultanement par une meme application sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le nombre maximum de sessions en lecture/criture pouvant tre ouvertes simultanment par une mme application sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie le nombre de sessions en lecture/criture que l'application courante a d'ouvertes sur la carte
	 * @return Nombre de sessions en lecture/criture que l'application courante a d'ouvertes sur la carte
	 * @throws ExceptionObjet 
	 */
	public long getNombreSessionsEnEcritureOuvertesParApplicationCourante() throws ExceptionObjet {
		
		if(info != null)
			return info.ulRwSessionCount;
		else {
			
			log.error("Les informations de la carte sont nulles, le nombre de sessions en lecture/ecriture que l'application courante a d'ouvertes sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le nombre de sessions en lecture/criture que l'application courante a d'ouvertes sur la carte est indisponible");
			
		}
		
	}	
	
	/**
	 * Renvoie la longueur maximale (en octets) du code porteur
	 * @return Longueur maximale (en octets) du code porteur
	 * @throws ExceptionObjet 
	 */
	public long getLongueurMaxPin() throws ExceptionObjet {
		
		if(info != null)
			return info.ulMaxPinLen;
		else {
			
			log.error("Les informations de la carte sont nulles, la longueur maximale (en octets) du code porteur de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "La longueur maximale (en octets) du code porteur de la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie la longueur minimale (en octets) du code porteur
	 * @return Longueur minimale (en octets) du code porteur
	 * @throws ExceptionObjet 
	 */
	public long getLongueurMinPin() throws ExceptionObjet {
		
		if(info != null)
			return info.ulMinPinLen;
		else {
			
			log.error("Les informations de la carte sont nulles, la longueur minimale (en octets) du code porteur de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "La longueur minimale (en octets) du code porteur de la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie la capacit mmoire totale (en octets) pour le stockage d'objets publics
	 * @return Capacit mmoire totale (en octets) pour le stockage d'objets publics
	 * @throws ExceptionObjet 
	 */
	public long getCapaciteMemoirePublicTotal() throws ExceptionObjet {
		if(info != null)
			return info.ulTotalPublicMemory;
		else {
			
			log.error("Les informations de la carte sont nulles, la capacite memoire totale (en octets) pour le stockage d'objets publics sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "La capacit mmoire totale (en octets) pour le stockage d'objets publics sur la carte est indisponible");
			
		}
	}

	/**
	 * Renvoie l'espace mmoire disponible (en octets) pour le stockage d'objets publics
	 * @return Espace mmoire disponible (en octets) pour le stockage d'objets publics
	 * @throws ExceptionObjet 
	 */
	public long getEspaceMemoirePublicDisponible() throws ExceptionObjet {
		
		if(info != null)
			return info.ulFreePublicMemory;
		else {
			
			log.error("Les informations de la carte sont nulles, l'espace memoire disponible (en octets) pour le stockage d'objets publics sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'espace mmoire disponible (en octets) pour le stockage d'objets publics sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Renvoie la capacit mmoire totale (en octets) pour le stockage d'objets privs
	 * @return Capacit mmoire totale (en octets) pour le stockage d'objets privs
	 * @throws ExceptionObjet 
	 */
	public long getCapaciteMemoirePriveTotal() throws ExceptionObjet {
		
		if(info != null)
			return info.ulTotalPrivateMemory;
		else {
			
			log.error("Les informations de la carte sont nulles, la capacite memoire totale (en octets) pour le stockage d'objets prives sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "La capacit mmoire totale (en octets) pour le stockage d'objets privs sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Renvoie l'espace mmoire disponible (en octets) pour le stockage d'objets privs
	 * @return Espace mmoire disponible (en octets) pour le stockage d'objets privs
	 * @throws ExceptionObjet 
	 */
	public long getEspaceMemoirePriveDisponible() throws ExceptionObjet {
		
		if(info != null)
			return info.ulFreePrivateMemory;
		else {
			
			log.error("Les informations de la carte sont nulles, l'espace memoire disponible (en octets) pour le stockage d'objets prives sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'espace mmoire disponible (en octets) pour le stockage d'objets privs sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie le numro de version matrielle de la carte
	 * @return Numro de version matrielle de la carte
	 * @throws ExceptionObjet 
	 */
	public String getVersionMaterielle() throws ExceptionObjet {
		
		if(info != null)
			return info.hardwareVersion.toString();
		else {
			
			log.error("Les informations de la carte sont nulles, le numero de version materielle de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le numro de version matrielle de la carte est indisponible");
			
		}

	}
	
	/**
	 * Renvoie le numro de version firmware de la carte
	 * @return Numro de version firmware de la carte
	 * @throws ExceptionObjet 
	 */
	public String getVersionFirmware() throws ExceptionObjet {
		
		if(info != null)
			return info.firmwareVersion.toString();
		else {
			
			log.error("Les informations de la carte sont nulles, le numero de version firmware de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "Le numro de version firmware de la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Renvoie la date de la carte
	 * @return Date de la carte
	 * @throws ExceptionObjet 
	 */
	public String getDate() throws ExceptionObjet {
		
		if(info != null)
			return new String(info.utcTime);
		else {
			
			log.error("Les informations de la carte sont nulles, la date de la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "La date de la carte est indisponible");
			
		}
	}

	/**
	 * La carte a-t-elle son propre gnrateur de nombre alatoire?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isGenerateurAleatoire() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_RNG)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * La carte est-elle protge en criture?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isProtegeEcriture() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_WRITE_PROTECTED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * La carte ncessite-t-elle une authentification?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isAuthentificationRequise() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_LOGIN_REQUIRED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * Le code porteur de l'utilisateur est-il initialis?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isPinInitialise() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_USER_PIN_INITIALIZED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * La sauvegarde russie de l'tat d'une opration cryptographique de session contient-elle toutes les cls ncessaires pour rtablir l'tat de la session?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isCleRestaurationRequise() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_RESTORE_KEY_NOT_NEEDED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * La carte possde-t-elle une horloge?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isHorloge() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_CLOCK_ON_TOKEN)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * La carte a-t-elle un mode d'authentification protg par o l'utilisateur peut s'authentifier sans passer de code porteur au travers d'une bibliothque cryptographique?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isAuthentificationProtegee() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_PROTECTED_AUTHENTICATION_PATH)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}
	
	
	/**
	 * La carte permet-elle l'excution d'oprations cryptographiques "dual" lors d'une mme session?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isDual() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_DUAL_CRYPTO_OPERATIONS)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}
	
	/**
	 * La carte est-elle initialise?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isInitialise() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_TOKEN_INITIALIZED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Un code porteur utilisateur incorrect a-t-il t saisi au moins une fois depuis la dernire authentification utilisateur russie?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isAuMoinsUnEchecPin() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_USER_PIN_COUNT_LOW)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * La prochaine saisie d'un code porteur utilisateur incorrect provoquera-t-elle le blocage du token?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isBloqueAuProchainEchecPin() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_USER_PIN_FINAL_TRY)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Le code porteur utilisateur est-il bloqu?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isPinBloque() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_USER_PIN_LOCKED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Le code porteur utilisateur doit-il tre chang?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isPinAChanger() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_USER_PIN_TO_BE_CHANGED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Un code porteur Security Officer(SO) incorrect a-t-il t saisi au moins une fois depuis la dernire authentification SO russie?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isAuMoinsUnEchecPuk() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_SO_PIN_COUNT_LOW)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * La prochaine saisie d'un code porteur SO incorrect provoquera-t-elle le blocage de la carte?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isBloqueAuProchainEchecPuk() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_SO_PIN_FINAL_TRY)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Le code porteur SO est-il bloqu?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isPukBloque() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_SO_PIN_LOCKED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}

	/**
	 * Le code porteur SO doit-il tre chang?
	 * @return Rponse
	 * @throws ExceptionObjet 
	 */
	public boolean isPukAChanger() throws ExceptionObjet {
		
		if(info != null)
			return (info.flags & PKCS11Constants.CKF_SO_PIN_TO_BE_CHANGED)!=0?true:false;
		else {
			
			log.error("Les informations de la session sont nulles, l'information sur la carte est indisponible");
			throw new ExceptionObjet(ExceptionProgrammeExemple.TYPE_OBJET_TOKEN, "L'information sur la carte est indisponible");
			
		}
		
	}
	
	@Override
	public String toString() {
		return getLibelle();
	}

}
